unit dmfload; (* Version 28. Mrz 2016 *)

interface

procedure open_dmf(name: string);
procedure close_dmf;
procedure load_dmf_infos;
procedure load_dmf_pattern(number: word);

type dmfevents = record
  note: byte;
  volume: byte;
  sampnr: byte;
end;

var
dmfstrucpat_pointer: pointer;

type dmfpattern = array[0..255, 0..15] of dmfevents;

var dmf_pattern: ^dmfpattern absolute dmfstrucpat_pointer;
    dmf_pat_speed: array[0..255] of byte;

var dmf_songname: string[30];
    dmf_composer: string[20];

    dmf_sequ_loopstart: word;
    dmf_sequ_loopend: word;
    dmf_sequencer: array[1..1024] of word;
    dmf_sequ_length: word;

    dmf_pattentries: word;
    dmf_maxusedtracks: byte;

    dmf_sample_entries: byte;
    dmf_sample_name: array[1..255] of string[30];

type dmf_smpinfo = record
     length: longint;
     loopstart: longint;
     loopend: longint;
     c3freq: word;
     volume: word;
     loop: bytebool;
end;

var  dmf_sample_info: array[1..255] of dmf_smpinfo;

type dmfpatternheader = record
     tracks: byte;
     ticksperbeat: byte;
     zeilen: byte;
end;

var dmf_pattern_header: array[1..1024] of dmfpatternheader;


var counter: array[0..32] of byte;

    dmf_samplestream_seek: longint;

    dmf_f: file;


implementation

type patternheader = record
  tracks: byte;
  ticksperbeat: byte;
  zeilen: word;
  datalength: longint;
end;

var
    pattern_header: array[1..1024] of patternheader;
    patterndata_seek: longint;

type dmf_smpi = record
     length: longint;
     loopstart: longint;
     loopend: longint;
     c3freq: word;
     volume: byte;
     typ: byte;
     libname: array[1..8] of char;
     filler: word;
     crc32: longint;
end;

var dmf_sample: array[1..255] of dmf_smpi;

var pattern_seek: array[1..1024] of longint;

    max_dmfpat_size: longint;
    dmfpat_pointer: pointer;

type patbuffer=array[0..65527] of byte;
var pat_buffer: ^patbuffer absolute dmfpat_pointer;

procedure open_dmf(name: string);
begin
  assign(dmf_f, name +'.dmf');
  reset(dmf_f, 1);
end;

procedure close_dmf;
begin
  close(dmf_f);
end;

procedure load_and_check_chunk(name: string);
var
  teststr: string[4];
begin
  teststr[0] := chr(4);

  blockread(dmf_f, teststr[1], 4);
(*  writeln(teststr); *)
  if not (teststr = name) then
  begin
    writeln(name + ' Chunk in der DMF-Datei nicht gefunden!');
    halt(1);
  end;
end;


procedure fill_sample_info;
var i: byte;
begin
  for i := 1 to dmf_sample_entries do
    with dmf_sample_info[i] do
    begin
      length := dmf_sample[i].length;
      loopstart := dmf_sample[i].loopstart;
      loopend := dmf_sample[i].loopend;
      c3freq := dmf_sample[i].c3freq;
      volume := dmf_sample[i].volume;

      if (dmf_sample[i].typ and 1) = 1 then
        loop := true
      else
        loop := false;

      if (dmf_sample[i].typ and $F) > 1  then begin
        writeln('Ungltige Samples im DMF - 16 Bit oder komprimiert!');
        halt(1);
      end;

    end;
end;


procedure fill_pattern_header;
var i: word;
begin
  for i := 1 to dmf_pattentries do
    with dmf_pattern_header[i] do
    begin
      tracks := pattern_header[i].tracks;
      ticksperbeat := pattern_header[i].ticksperbeat shr 4;
      zeilen := pattern_header[i].zeilen;
      if pattern_header[i].zeilen > 256 then begin
        writeln('Pattern berschreitet 256 Zeilen!');
        halt(1);
      end;
    end;
end;

procedure alloc_pattmem;
begin
  getmem(dmfstrucpat_pointer, 6144*2);

  if max_dmfpat_size > 65528 then
  begin
    writeln('DMF Pattern zu gro fr Speicher!');
    halt(1);
  end;
  getmem(dmfpat_pointer, max_dmfpat_size);
end;


procedure load_dmf_infos;
var
  i, ii: byte;
  iii: word;
  blocklength: longint;
  smpi_seek: longint;
begin

  load_and_check_chunk('DDMF');

  seek(dmf_f, $D);
  blockread(dmf_f, dmf_songname[1], 30);
  i := 30;
  while (dmf_songname[i] = chr(0)) and (i > 0) do dec(i);
  dmf_songname[0] := chr(i);
  blockread(dmf_f, dmf_composer[1], 20);
  i := 20;
  while (dmf_composer[i] = chr(0)) and (i > 0) do dec(i);
  dmf_songname[0] := chr(i);
  { Titel und Composer eingelesen und Strings zurechtgestutzt }

  seek(dmf_f, filepos(dmf_f) + 3);
  load_and_check_chunk('CMSG');
  blockread(dmf_f, blocklength, 4); { CMSG Block berspringen }

  seek(dmf_f, filepos(dmf_f) + blocklength);
  load_and_check_chunk('SEQU');

  blockread(dmf_f, blocklength, 4);
  dmf_sequ_length := blocklength div 2 - 2;
  blockread(dmf_f, dmf_sequ_loopstart, 2);
  blockread(dmf_f, dmf_sequ_loopend, 2);
  blockread(dmf_f, dmf_sequencer, blocklength - 4); { Sequencer einlesen }

  load_and_check_chunk('PATT');

  blockread(dmf_f, blocklength, 4);
  smpi_seek := filepos(dmf_f) + blocklength;
  blockread(dmf_f, dmf_pattentries, 2);
  blockread(dmf_f, dmf_maxusedtracks, 1);
  patterndata_seek := filepos(dmf_f);

  seek(dmf_f, smpi_seek);
  load_and_check_chunk('SMPI');
  seek(dmf_f, filepos(dmf_f) + 4);
  blockread(dmf_f, dmf_sample_entries, 1);

  for i := 1 to dmf_sample_entries do
  begin
    blockread(dmf_f, ii, 1);
    dmf_sample_name[i][0] := chr(ii);
    blockread(dmf_f, dmf_sample_name[i][1], ii);

    blockread(dmf_f, dmf_sample[i], 30);
  end;

  fill_sample_info;

  load_and_check_chunk('SMPD');
  dmf_samplestream_seek := filepos(dmf_f) + 4; { hinter SMPD und Blocklength }

  seek(dmf_f, patterndata_seek);

  max_dmfpat_size := 0;
  for iii := 1 to dmf_pattentries do
  begin
    blockread(dmf_f, pattern_header[iii], 8);
    pattern_seek[iii] := filepos(dmf_f);

    seek(dmf_f, filepos(dmf_f) + pattern_header[iii].datalength);
    { zum nchsten Pattern springen bzw. die Daten dieses Patterns berspr. }

(*    writeln('Pattern #', iii, '   Datengre: ', pattern_header[iii].datalength); *)
    if pattern_header[iii].datalength > max_dmfpat_size then
      max_dmfpat_size := pattern_header[iii].datalength;
  end;

  fill_pattern_header;

(*  writeln('maxpatsize: ', max_dmfpat_size); *)

  alloc_pattmem; { Hier schon den Pufferspeicher fr Patterns einrichten }

end;


procedure load_dmf_pattern(number: word);
  var zeile, track: byte;
  readpos: word;
  globaltrack_event: byte;
  info: byte;
begin

  seek(dmf_f, pattern_seek[number]);
  blockread(dmf_f, pat_buffer^[0], pattern_header[number].datalength);

  readpos := 0;
  for track := 0 to pattern_header[number].tracks do counter[track] := 0;
  fillchar (dmf_pattern^, 6144*2, 0);
  fillchar (dmf_pat_speed, 256, 0);


  for zeile := 0 to pattern_header[number].zeilen -1 do
  begin

    {GLOBAL TRACK}
    if counter[0] = 0 then
    begin
      globaltrack_event := pat_buffer^[readpos]; { INFO }
      inc(readpos);
      if (globaltrack_event and 128) = 128 then { PACK }
      begin
        counter[0] := pat_buffer^[readpos];
        inc(readpos);
      end;
      globaltrack_event := globaltrack_event and $3F;
      if globaltrack_event > 0 then { DATA }
      begin
        dmf_pat_speed[zeile] := pat_buffer^[readpos];
        inc(readpos);
        if not (globaltrack_event = 2) then dmf_pat_speed[zeile] := 0;
        { Alle Effekte ausser BPM set ignorieren }
      end;
    end
    else
      dec(counter[0]);
    { GLOBAL TRACK ENDE }

    { NORMALE TRACKS }
    for track := 1 to pattern_header[number].tracks do
    begin
      if counter[track] = 0 then
      begin

        info := pat_buffer^[readpos];
        inc(readpos);
        if (info and 128) = 128 then { COUNTER }
        begin
          counter[track] := pat_buffer^[readpos];
          inc(readpos);
        end;
        if (info and 64) = 64 then { INSTRUMENT }
        begin
          if track < 17 then { nur bis Track 16 ! }
            dmf_pattern^[zeile, track-1].sampnr := pat_buffer^[readpos];
          inc(readpos);
        end;
        if (info and 32) = 32 then { NOTE }
        begin
          if track < 17 then { nur bis Track 16 ! }
            dmf_pattern^[zeile, track-1].note := pat_buffer^[readpos];
          inc(readpos);
        end;
        if (info and 16) = 16 then { VOLUME }
        begin
          if track < 17 then { nur bis Track 16 ! }
            dmf_pattern^[zeile, track-1].volume := pat_buffer^[readpos];
          inc(readpos);
        end;
        if (info and 8) = 8 then { INSTRUMENT EFFECT UEBERSPRINGEN! }
          inc(readpos, 2);
        if (info and 4) = 4 then { NOTE EFFECT UEBERSPRINGEN! }
          inc(readpos, 2);
        if (info and 2) = 2 then { VOLUME EFFECT UEBERSPRINGEN! }
          inc(readpos, 2);
      end
      else
        dec(counter[track]);
    end;
    { NORMALE TRACKS ENDE }
  end; { for zeile }

end;


end.