uses dmfload, bitwrite;

type zsm_header = record
  kennung: longint; (* $4D54535A "ZSTM" *)
  version: byte;
  titel: array[0..19] of char;
  composer: array[0..19] of char;
  sample_entries: byte;
  sequencer_entries: word;
  sequencer_loopstart: word;
  sequencer_loopend: word;
  patterndata_size: word;
end;


type sampleinfo = record
  length: word;
  loopstart: word;
  marker: byte;
  c3freq: word;
  volume: byte;
end;


type pattern_header = record
  size: word;
  rows: byte;
  samples_per_row: word;
  tracks_active: word;
end;

type fixkomma = record
  vor: byte;
  nach: word;
end;

var

  buffer: pointer;

  logfile: text;

  patterns_pointer: pointer;

  zsmheader: zsm_header;

  bitsneeded: array[0..256] of byte;

  lastpos_patdata: word;

  pattern_used: array[1..1024] of bytebool;
  patternheader: array[1..1024] of pattern_header;

  pattern_addr: array[1..1024] of word;


  pattern_bpm: byte;

  sequencer: array [1..1024] of word;


  pitch_tables: array[1..63, 0..31] of fixkomma;
  pitch_table_length: array[1..63] of byte;

  note_ref_table: array[1..63, 0..31] of byte;


  freqtable: array[1..108] of real;

  volconvert: array[0..255] of byte;

  patterndata_count: word;

  smpinfo: array[1..63] of sampleinfo;

  do_smpnr_table: array[0..15] of bytebool;
  do_event_repeat: array[0..15] of bytebool;

  sampnr_table_len: array[0..15] of byte;
  sampnr_table: array[0..15, 0..63] of byte;

  volume_table_len: array[0..15] of byte;
  volume_table: array[0..15, 0..7] of byte;

  use_emptyrow_flags: bytebool;

  zsmname: string;


procedure fill_volconvert;
var
  i: byte;
begin
  for i := 192 to 255 do volconvert[i] := 7;
  for i := 96 to 191 do volconvert[i] := 6;
  for i := 48 to 95 do volconvert[i] := 5;
  for i := 24 to 47 do volconvert[i] := 4;
  for i := 12 to 23 do volconvert[i] := 3;
  for i := 6 to 11 do volconvert[i] := 2;
  for i := 3 to 5 do volconvert[i] := 1;
  for i := 1 to 2 do volconvert[i] := 0;
end;

procedure fill_freqtable;
  var i, ii: byte;
begin
  freqtable[97]  := 64.0 / 2;
  freqtable[98]  := 67.80563803899 / 2;
  freqtable[99]  := 71.83757109180 / 2;
  freqtable[100] := 76.10925536017 / 2;
  freqtable[101] := 80.63494719327 / 2;
  freqtable[102] := 85.42975066688 / 2;
  freqtable[103] := 90.50966799188 / 2;
  freqtable[104] := 95.89165292011 / 2;
  freqtable[105] := 101.59366732596 / 2;
  freqtable[106] := 107.63474115248 / 2;
  freqtable[107] := 114.03503592196 / 2;
  freqtable[108] := 120.81591202326 / 2;

  for i := 0 to 7 do
    for ii := 1 to 12 do
    freqtable[i * 12 + ii] := freqtable[96 + ii] / (256 shr i);

end;

procedure fill_bitsneeded_table; (* wieviel bit gebraucht werden *)
  var i: word;                   (* um entsprechend lange Tabelle *)
begin                            (* zu indizieren *)
  bitsneeded[0] := 0;
  bitsneeded[1] := 0;
  bitsneeded[2] := 1;
  bitsneeded[3] := 2;
  bitsneeded[4] := 2;
  for i := 5 to 8 do
    bitsneeded[i] := 3;
  for i := 9 to 16 do
    bitsneeded[i] := 4;
  for i := 17 to 32 do
    bitsneeded[i] := 5;
  for i := 33 to 64 do
    bitsneeded[i] := 6;
  for i := 65 to 128 do
    bitsneeded[i] := 7;
  for i := 129 to 256 do
    bitsneeded[i] := 8;
end;


procedure set_up_patternstream;
begin
  getmem(patterns_pointer, 65528);
  set_bitpointer(patterns_pointer);
end;


procedure make_marker(sample: byte);
begin

  with smpinfo[sample] do
  begin
    if dmf_sample_name[sample] = 'bassdrum' then marker := $80;
    if dmf_sample_name[sample] = 'snare' then marker := $81;
    if dmf_sample_name[sample] = 'hihat' then marker := $82;
    if dmf_sample_name[sample] = 'cymbal' then marker := $83;
    if dmf_sample_name[sample] = 'bass' then marker := $90;
  end;

end;




procedure make_sample_infos;
  var
    sample: byte;
begin

  for sample := 1 to dmf_sample_entries do
  begin

    smpinfo[sample].marker := 0;
    make_marker(sample);

    if dmf_sample_info[sample].length > 65528 then
    begin
      writeln('Error: Sample #', sample, ' > 65528 Byte');
      halt(1);
    end;

    smpinfo[sample].length := dmf_sample_info[sample].length;

    if dmf_sample_info[sample].loop then
    begin
      smpinfo[sample].length := dmf_sample_info[sample].loopend+1;
      if smpinfo[sample].length > dmf_sample_info[sample].length then
        smpinfo[sample].length := dmf_sample_info[sample].length;
      smpinfo[sample].loopstart := dmf_sample_info[sample].loopstart;
    end
    else
      smpinfo[sample].loopstart := smpinfo[sample].length;


    smpinfo[sample].c3freq := dmf_sample_info[sample].c3freq;
    smpinfo[sample].volume := dmf_sample_info[sample].volume;

  end;
end;





procedure flag_used_patterns;
  var i: word;
begin
  fillchar(pattern_used[1], 1024, 0);
  for i := 1 to dmf_sequ_length do
    pattern_used[dmf_sequencer[i]+1] := true;
  (* Sequencer in DMF indiziert Patterns ab Null *)
  (* Wir wollen aber Patterns ab 1, deshalb + 1 *)
end;


procedure make_pattern_header(pattern: word);
  var
    row: byte;
    track: word;
begin

  patternheader[pattern].size := 0; (* Wird spaeter gefuellt *)
                                    (* wird nicht in Datei geschrieben *)
                                    (* nur fuer den Log *)
  patternheader[pattern].rows
    := dmf_pattern_header[pattern].zeilen - 1;

  {
  if dmf_pattern_header[pattern].ticksperbeat = 0 then
    dmf_pattern_header[pattern].ticksperbeat := 4;
  if pattern_bpm = 0 then pattern_bpm := 125;
  }
  writeln(pattern_bpm);
  patternheader[pattern].samples_per_row := 0;
  if pattern_bpm * dmf_pattern_header[pattern].ticksperbeat > 0 then
  patternheader[pattern].samples_per_row
    := (60 * 22050 {mixfrequenz}) div (pattern_bpm * dmf_pattern_header[pattern].ticksperbeat);

  patternheader[pattern].tracks_active := 0;

  for track := 0 to 15 do
    for row := 0 to patternheader[pattern].rows do
      if (dmf_pattern^[row, track].note > 0)
        or (dmf_pattern^[row, track].volume > 0) then
          patternheader[pattern].tracks_active
            := patternheader[pattern].tracks_active or (1 shl track);

end;


procedure update_pitchtable(sample, note: byte);
  var
    i: byte;
    new: bytebool;
begin
  i := 0;
  new := true;
  for i := 1 to pitch_table_length[sample] do
    if pitch_tables[sample, i-1].vor = note then
      new := false;

   if new then begin
     inc(pitch_table_length[sample]);

     if i > 32 then begin
       writeln('Converter Error: too many different Notes with Sample #', sample);
       halt(1);
     end;

     pitch_tables[sample, pitch_table_length[sample]-1].vor := note;

     note_ref_table[sample, pitch_table_length[sample]-1] := note;

   end;

(*************************************)

end;

{
procedure fix_note_without_sampnr(pattern: word; track: byte);
  var
    row: byte;
    smp_set: byte;
begin
  smp_set := 0;
  for row := 0 to patternheader[pattern].rows do
    with dmf_pattern^[row, track] do
    begin
      if sampnr > 0 then smp_set := sampnr;
      if (note > 0) and (note < 255) and (sampnr = 0) then
        sampnr := smp_set;
    end;
end;
}



procedure make_pitch_tables;
  var
    pattern: word;
    sample: byte;
    track, row: byte;
    i: byte;
begin

  fillchar(pitch_tables[1, 0], 63*32*3, 0);
  fillchar(pitch_table_length[1], 63, 0);

  for sample := 1 to dmf_sample_entries do
    for pattern := 1 to dmf_pattentries do
      if pattern_used[pattern] then
      begin
        load_dmf_pattern(pattern);

        for track := 0 to 15 do
          for row := 0 to dmf_pattern_header[pattern].zeilen - 1 do
          if (dmf_pattern^[row, track].note > 0) and
            (dmf_pattern^[row, track].note < 255) and
            (dmf_pattern^[row, track].sampnr = sample) then
            update_pitchtable(sample, dmf_pattern^[row, track].note);
      end;
(* erst nur die Note einspeichern, spter die konkreten Periods ausrechnen *)

end;



procedure make_periods;
  var
    sample: byte;
    note, i: byte;
    vorcheck: word;
    fullstep: real;
begin

  for sample := 1 to dmf_sample_entries do
    if pitch_table_length[sample] > 0 then
      for i := 0 to pitch_table_length[sample] - 1 do
      begin

        note := pitch_tables[sample, i].vor;

        fullstep := dmf_sample_info[sample].c3freq / 22050{mixfrequenz} * freqtable[note];
        vorcheck := trunc(fullstep);
        if vorcheck > 255 then
        begin
          writeln('Error: Sample Step-Speed > 255');
          halt(1);
        end;

        pitch_tables[sample, i].vor := vorcheck;
        pitch_tables[sample, i].nach :=
          trunc((fullstep - vorcheck) * 65536);

        {
        write('Sample #', sample, ' Period #', i+1, ': ');
        writeln(pitch_tables[sample, i].vor, '.', pitch_tables[sample, i].nach);
        }
      end;

end;





procedure check_repeat_and_samplenr_optimum(pattern: word; track: byte);
var
  row, i: byte;
  bitcount: array[1..4] of word;
  sampnr_table: array[0..63] of byte;
  tbl_len: byte;
  in_table: bytebool;
  last_sampnr: byte;
  last_note: byte;
  kleiner1, kleiner2: byte;
  sampnr: byte;
  smpentries: byte;
  noteoff: bytebool;
begin
  fillchar(bitcount[1], 8, 0);
  smpentries := dmf_sample_entries;


  (* Fall1: weder Repeat noch SampleNr-Tabelle *)
  for row := 0 to patternheader[pattern].rows do
  begin
    noteoff := dmf_pattern^[row, track].note = 255;
    sampnr := dmf_pattern^[row, track].sampnr;

    if noteoff then
      inc(bitcount[1], bitsneeded[smpentries+1])
    else
    if sampnr > 0 then
    begin
        inc(bitcount[1], bitsneeded[smpentries+1]);
        inc(bitcount[1], bitsneeded[pitch_table_length[sampnr]]);
        (* Auch Volume Tabelle beruecksichtigen *)
        inc(bitcount[1], bitsneeded[volume_table_len[track]]);
    end;

  end;



  (* Fall2: SampNr-Tabelle *)
  tbl_len := 0;

  for row := 0 to patternheader[pattern].rows do
    if dmf_pattern^[row, track].note = 255 then
    begin
      tbl_len := 1;
      sampnr_table[0] := 0; (* Note Off einfgen, falls im Track *)
    end;

  for row := 0 to patternheader[pattern].rows do
  begin
    sampnr := dmf_pattern^[row, track].sampnr;
    if sampnr > 0 then
    begin
      in_table := false;
      for i := 1 to tbl_len do
        if sampnr = sampnr_table[i-1]
          then in_table := true;
      if not in_table then
      begin
        sampnr_table[tbl_len] := sampnr;
        inc(tbl_len);
      end;
    end;
  end;
  for row := 0 to patternheader[pattern].rows do
  begin
    noteoff := dmf_pattern^[row, track].note = 255;
    sampnr := dmf_pattern^[row, track].sampnr;

    if noteoff then
      inc(bitcount[2], bitsneeded[tbl_len])
    else
    if sampnr > 0 then
    begin
      inc(bitcount[2], bitsneeded[tbl_len]);
      inc(bitcount[2], bitsneeded[pitch_table_length[sampnr]]);
      (* Auch Volume Tabelle beruecksichtigen *)
      inc(bitcount[2], bitsneeded[volume_table_len[track]]);
    end;
  end;
  inc(bitcount[2], bitsneeded[smpentries+1] * tbl_len);
  if tbl_len > 16 then bitcount[2] := $FFFF; (* keine Tabellen > 16 *)
  {
  writeln(tbl_len);
  writeln(bitsneeded[dmf_sample_entries] * tbl_len);
  }

  (* Fall3: Nur Repeat *)
  last_sampnr := 0;
  last_note := 0;
  for row := 0 to patternheader[pattern].rows do
  begin
    noteoff := dmf_pattern^[row, track].note = 255;
    sampnr := dmf_pattern^[row, track].sampnr;
    if (sampnr > 0) or noteoff then
      if (sampnr = last_sampnr)
        and (dmf_pattern^[row, track].note = last_note) then
      begin
        inc(bitcount[3], 1);
        (* Auch Volume Tabelle beruecksichtigen *)
        if not noteoff then
          inc(bitcount[3], bitsneeded[volume_table_len[track]]);
      end
      else
      begin
        inc(bitcount[3], 1);
        inc(bitcount[3], bitsneeded[smpentries+1]);
        if not noteoff then
          inc(bitcount[3], bitsneeded[pitch_table_length[sampnr]]);
        (* Auch Volume Tabelle beruecksichtigen *)
        if not noteoff then
          inc(bitcount[3], bitsneeded[volume_table_len[track]]);
        last_sampnr := sampnr;
        last_note := dmf_pattern^[row, track].note;
      end;
  end;

  (* Fall4: SampNr-Tabelle und Repeat *)
  last_sampnr := 0;
  last_note := 0;
  tbl_len := 0;

  for row := 0 to patternheader[pattern].rows do
    if dmf_pattern^[row, track].note = 255 then
    begin
      tbl_len := 1;
      sampnr_table[0] := 0; (* Note Off einfgen, falls im Track *)
    end;

  for row := 0 to patternheader[pattern].rows do
  begin
    sampnr := dmf_pattern^[row, track].sampnr;
    if sampnr > 0 then
    begin
      in_table := false;
      for i := 1 to tbl_len do
        if sampnr = sampnr_table[i-1]
          then in_table := true;
      if not in_table then
      begin
        sampnr_table[tbl_len] := sampnr;
        inc(tbl_len);
      end;
    end;
  end;

  for row := 0 to patternheader[pattern].rows do
  begin
    noteoff := dmf_pattern^[row, track].note = 255;
    sampnr := dmf_pattern^[row, track].sampnr;
    if sampnr > 0 then
      if (sampnr = last_sampnr)
        and (dmf_pattern^[row, track].note = last_note) then
      begin
        inc(bitcount[4], 1);
        (* Auch Volume Tabelle beruecksichtigen *)
        if not noteoff then
          inc(bitcount[4], bitsneeded[volume_table_len[track]]);
      end
      else
      begin
        inc(bitcount[4], 1);
        inc(bitcount[4], bitsneeded[tbl_len]);
        if not noteoff then
          inc(bitcount[4], bitsneeded[pitch_table_length[sampnr]]);
        (* Auch Volume Tabelle beruecksichtigen *)
        if not noteoff then
          inc(bitcount[4], bitsneeded[volume_table_len[track]]);
        last_sampnr := sampnr;
        last_note := dmf_pattern^[row, track].note;
      end;
  end;
  inc(bitcount[4], bitsneeded[smpentries+1] * tbl_len);
  if tbl_len > 16 then bitcount[4] := $FFFF; (* keine Tabellen > 16 *)


  if bitcount[1] < bitcount[2] then kleiner1 := 1 else kleiner1 := 2;
  if bitcount[3] < bitcount[4] then kleiner2 := 3 else kleiner2 := 4;
  if bitcount[kleiner1] > bitcount[kleiner2] then kleiner1 := kleiner2;

  {
  writeln('Pattern #', pattern, '  Track #', track, ':');
  writeln;
  writeln('No Repeat, no SampNr Table: ', bitcount[1], ' Bit');
  writeln('Only SampNr Table: ', bitcount[2], ' Bit');
  writeln('Only Repeat: ', bitcount[3], ' Bit');
  writeln('Repeat and SampNr Table: ', bitcount[4], ' Bit');

  writeln('Chosen Method: ', kleiner1);
  }
  if kleiner1 = 1 then
  begin
    do_smpnr_table[track] := false;
    do_event_repeat[track] := false;
  end;

  if kleiner1 = 2 then
  begin
    do_smpnr_table[track] := true;
    do_event_repeat[track] := false;
  end;

  if kleiner1 = 3 then
  begin
    do_smpnr_table[track] := false;
    do_event_repeat[track] := true;
  end;

  if kleiner1 = 4 then
  begin
    do_smpnr_table[track] := true;
    do_event_repeat[track] := true;
  end;

end;


procedure make_sampnr_table(pattern: word; track: byte);
  var
    row, i: byte;
    in_table: bytebool;
    tbl_len: byte;
begin
  tbl_len := 0;

  for row := 0 to patternheader[pattern].rows do
  begin


    in_table := false;

    if dmf_pattern^[row, track].note = 255 then
    begin

      for i := 1 to tbl_len do
      if sampnr_table[track, i-1] = 0 then
        in_table := true;
      if not in_table then
      begin
        sampnr_table[track, tbl_len] := 0;
        inc(tbl_len);
      end;

    end
    else
    begin


      for i := 1 to tbl_len do
        if dmf_pattern^[row, track].sampnr = sampnr_table[track, i-1] then
          in_table := true;

      if (not in_table) and (dmf_pattern^[row, track].sampnr > 0) then
      begin
        sampnr_table[track, tbl_len] := dmf_pattern^[row, track].sampnr;
        inc(tbl_len);
      end;
    end;


  end;

  sampnr_table_len[track] := tbl_len;

end;



procedure make_volume_table(pattern: word; track: byte);
  var
    row, i: byte;
    in_table: bytebool;
    tbl_len: byte;
    wert: byte;
begin
  tbl_len := 0;

  for row := 0 to patternheader[pattern].rows do
  begin

    (* In den Patterns wird immer ein Wert gebraucht *)
    (* Im Zweifelsfall die Standardlautstaerke des Samples *)
    if dmf_pattern^[row, track].volume = 0 then
      if dmf_pattern^[row, track].sampnr > 0 then
      dmf_pattern^[row, track].volume
      := smpinfo[dmf_pattern^[row, track].sampnr].volume;


    wert := $FF;
    if dmf_pattern^[row, track].volume > 0 then
     wert := volconvert[dmf_pattern^[row, track].volume]
    else
      if dmf_pattern^[row, track].sampnr > 0 then
        wert:=volconvert[smpinfo[dmf_pattern^[row, track].sampnr].volume];
    if not (wert = $FF) then (* Wenn ein Wert vorliegt *)
    begin

      in_table := false;
      for i := 1 to tbl_len do
        if wert = volume_table[track, i-1] then in_table := true;

      if not in_table then
      begin
        volume_table[track, tbl_len] := wert;
        inc(tbl_len);
      end;

    end;


  end;

  volume_table_len[track] := tbl_len;

end;




function sampnrtbl_pos(track, wert: byte): byte;
  var
    i: byte;
begin

  i := 0;
  (* Tabelle enthlt Samplenr - 1 *)
  while not (sampnr_table[track, i] = wert) do inc(i);
  sampnrtbl_pos := i;

end;


function volumetbl_pos(track, wert: byte): byte;
  var
    i: byte;
begin

  i := 0;
  while not (volume_table[track, i] = wert) do inc(i);
  volumetbl_pos := i;

end;


function notetbl_pos(sample, wert: byte): byte;
  var
    i: byte;
begin

  i := 0;
  while not (note_ref_table[sample, i] = wert) do inc(i);
  notetbl_pos := i;

end;


function empty_row(pattern: word; row: byte): bytebool;
  var
    track: word;
    empty: bytebool;
begin
  empty := true;
  for track := 0 to 15 do
  if patternheader[pattern].tracks_active and (1 shl track) > 0 then
    if (dmf_pattern^[row, track].sampnr > 0)
      or (dmf_pattern^[row, track].note = 255)
        or (dmf_pattern^[row, track].volume > 0)
          then empty := false;
  empty_row := empty;
end;


{
function second_notetable_effective(pattern, track): bytebool;
  var
    bitcount1: longint;
    row: byte;
begin
  bitcount1 := 0;
  for i := 0 to patternheader[pattern].rows do
    if dmf_pattern^[row, track].sampnr > 0 then
       inc(bitcount,
        bitsneeded[pitch_table_length[dmf_pattern^[row,track].sampnr]]);


end;
}


procedure check_emptyrows_effective(pattern, numbchannels: byte); (* ob es sich lohnt, Leerzeilen zu flaggen *)
  var row: byte;
  bitcount1, bitcount2: word;
begin

  bitcount1 := 0; (* Counter fuer mit Leerzeilen *)
  bitcount2 := 0; (* Counter fuer ohne Leerzeilen *)

  for row := 0 to patternheader[pattern].rows do
    if empty_row(pattern, row) then
    begin
      inc(bitcount1);              (* Nur das Leerzeilen Flag *)
      inc(bitcount2, numbchannels);(* kein Leer-Flag, aber Channel-Flags *)
    end
    else
    begin
      inc(bitcount1); (* zum einen das Leerzeilen Flag *)
      inc(bitcount1, numbchannels); (* aber auch die Channel-Flags *)
      inc(bitcount2, numbchannels); (* wiederum nur die Channel-Flags *)
    end;

  (* Leerzeilen verwenden wenn bitcount1 < bitcount2 *)
  use_emptyrow_flags := bitcount1 < bitcount2;

end;



function check_if_volume_only_exist(pattern: word; track: byte): bytebool;
  var
    row: byte;
    exist: bytebool;
begin
  exist := false;

  for row := 0 to patternheader[pattern].rows do
    with dmf_pattern^[row, track] do
    if (volume > 0) and (note = 0) and (sampnr = 0) then
      exist := true;

  check_if_volume_only_exist := exist;

end;

function track_has_sampnr(pattern: word; track: byte): bytebool;
  var
    smpnr_there: bytebool;
    row: byte;
begin

  smpnr_there := false;
  for row := 0 to patternheader[pattern].rows do
    if dmf_pattern^[row, track].sampnr > 0 then
      smpnr_there := true;
  track_has_sampnr := smpnr_there;

end;



procedure encode_pattern(pattern: word);
  var
    i, row, emptycount: byte;
    track: word;
    trk_active: array[0..15] of bytebool;
    num_tracks_active: byte;
    event, skip: bytebool;
    sampnr, volume, note: byte;
    lastnote, lastsamp: array[0..15] of byte;
    skipcount: byte;
    noteoff: bytebool;
    volume_only_exist: array[0..15] of bytebool;
    label skip2, skip_onlyvol;
begin

  load_dmf_pattern(pattern);

  i := 0;
  while (dmf_pat_speed[i] = 0) and (i < 255) do inc(i);
  {
  if dmf_pat_speed[i] = 0 then
    pattern_bpm := 125 else pattern_bpm := dmf_pat_speed[i];
  } (* 0 BPM zulassen, der Player nimmt dann Standard Werte *)
    (* oder welche vom vorangegangenen Pattern *)
  (* gegebene BPM aber bernehmen *)

  if dmf_pat_speed[i] > 0 then
    pattern_bpm := dmf_pat_speed[i];

  make_pattern_header(pattern);

    (* Beste Methoden je Track fr Kombi Sampnr / Repeat ermmitteln *)
    for track := 0 to 15 do
      check_repeat_and_samplenr_optimum(pattern, track);

  num_tracks_active := 0;
  for track := 0 to 15 do
    if patternheader[pattern].tracks_active and (1 shl track) > 0 then
    begin
      trk_active[track] := true;
      inc(num_tracks_active)
    end
    else
      trk_active[track] := false;


  for track := 0 to 15 do
    writeln(logfile, 'Track ', track, ' active: ', trk_active[track]);

  for track := 0 to 15 do
    volume_only_exist[track] := check_if_volume_only_exist(pattern, track);

  for track := 0 to 15 do
    if volume_only_exist[track] then
      writeln(logfile, 'Volume only existing on track ', track, ' Pattern ', pattern);

  check_emptyrows_effective(pattern, num_tracks_active);

  (* SampNr Tabellen machen *)
  for track := 0 to 15 do
    if trk_active[track] then
      if track_has_sampnr(pattern, track) then
        make_sampnr_table(pattern, track);

  (* Volume Tabellen machen *)
  for track := 0 to 15 do
    if trk_active[track] then
      make_volume_table(pattern, track);



   (* Hier fangen die Patterndaten an *)

   (* Adresse merken fr den Sequencer *)
   pattern_addr[pattern] := bitwrite_position div 8;

   lastpos_patdata := bitwrite_position div 8;

   write_bits(patternheader[pattern].rows, 8);
   write_bits(patternheader[pattern].samples_per_row and $FF, 8);
   write_bits(patternheader[pattern].samples_per_row shr 8, 8);
   write_bits(patternheader[pattern].tracks_active and $FF, 8);
   write_bits(patternheader[pattern].tracks_active shr 8, 8);


   (* Tabellen schreiben *)
   for track := 0 to 15 do
     if trk_active[track] then
     begin
       (* SampNr *)
       if do_smpnr_table[track] and track_has_sampnr(pattern, track) then
       begin
         write_bits(1, 1);

         writeln(logfile, 'Track ', track, ' SampNr-Table Len: ', sampnr_table_len[track]);

         write_bits(sampnr_table_len[track]-1, bitsneeded[dmf_sample_entries+1]);
         {write_bits(sampnr_table_len[track], bitsneeded[dmf_sample_entries+1]);}
         for i := 1 to sampnr_table_len[track] do
         begin
           write_bits(sampnr_table[track, i-1], bitsneeded[dmf_sample_entries+1]);
           write(logfile, 'Entry ', i, ': ', sampnr_table[track, i-1]);
         end;
         writeln(logfile);
         writeln(logfile);
       end
       else
         write_bits(0, 1);

       (* Volume *)

       if volume_table_len[track] = 0 then
       begin
         inc(volume_table_len[track]); (* Bugfix fuer Kanaele mit nur Noteoff *)
         volume_table[track, 0] := 7;
       end;


       write_bits(volume_table_len[track] - 1, 3);
         (* Volume hat immer mind. 1 Eintrag, daher 0 -> 1 Eintrag *)
         (* und alles, 1-8 bzw. 0-7 ist mit 3 Bit abgedeckt *)

       for i := 0 to volume_table_len[track] - 1 do
         write_bits(volume_table[track, i], 3);

       writeln(logfile, 'vol-tbllen: ', volume_table_len[track]);

       (* Repeat *)
       if do_event_repeat[track] then write_bits(1, 1) else write_bits(0, 1);

       if do_event_repeat[track] then
         writeln(logfile, 'Event Repeat in Pattern ',pattern,' on Track ', track);

       if volume_only_exist[track] then write_bits(1, 1) else write_bits(0, 1);

       {writeln(logfile, 'Pattern ', pattern, 'Track ', track,' has volume only.');}

     end;

     if use_emptyrow_flags then write_bits(1, 1) else write_bits(0, 1);

     if use_emptyrow_flags then writeln(logfile, 'Pattern ', pattern, ' uses Empty Row Flags.');

  (* Zeilen schreiben *)
  for track := 0 to 15 do
  begin
    lastnote[track] := 254; (* Note die niemals vorkommt *)
    lastsamp[track] := 255; (* Sample das niemals vorkommt *)
  end;

  for row := 0 to patternheader[pattern].rows do
  begin

    if use_emptyrow_flags then
      if empty_row(pattern, row) then
      begin
        write_bits(0, 1);
        goto skip2;
      end
      else
        write_bits(1, 1);

    for track := 0 to 15 do if trk_active[track] then
    begin

      event := false;
      sampnr := dmf_pattern^[row, track].sampnr;
      note := dmf_pattern^[row, track].note;
      volume := dmf_pattern^[row, track].volume;
      noteoff := note = 255;
      if noteoff then sampnr := 0;

      if (sampnr > 0) or noteoff then event := true;
      if volume_only_exist[track] then
        if volume > 0 then event := true; (* Volume auch alleine *)



      (* Event Flag pro Kanal *)
      if event then write_bits(1, 1) else write_bits(0, 1);

      if event then (* Wenn Event vorhanden *)
      begin

        skip := false;


      (* Volume only Flag *)
      (* Volume wird bei jedem Event eingelesen *)

      if volume_only_exist[track] then
      begin

        if (sampnr = 0) and (note = 0) then
        begin
          write_bits(0, 1);
          goto skip_onlyvol;
        end
        else
          write_bits(1, 1);

      end;


        (* Event Repeat *)

        if do_event_repeat[track] then
          if (sampnr = lastsamp[track]) and (note = lastnote[track]) then
          begin
            write_bits(1, 1);
            skip := true;
          end
          else
          begin
            write_bits(0, 1);
              if not noteoff then
              begin
                lastsamp[track] := sampnr;
                lastnote[track] := note;
              end;
          end;


        if not skip then
        begin

          if sampnr_table_len[track] = 0 then
            write_bits(sampnr, bitsneeded[dmf_sample_entries+1]);

          if sampnr_table_len[track] > 0 then
          begin

            (* Sampnr *)
            if do_smpnr_table[track] and track_has_sampnr(pattern, track) then
              write_bits(sampnrtbl_pos(track, sampnr), bitsneeded[sampnr_table_len[track]])
            else
              write_bits(sampnr, bitsneeded[dmf_sample_entries+1]);

            {
            if do_smpnr_table[track] then
              writeln(logfile, 'Track ', track, ', SmpNr from Table: ',
                sampnr_table[track, sampnrtbl_pos(track, sampnr)], '  Bits used: ', bitsneeded[sampnr_table_len[track]]);
            }
            (* Note *)
            if note = 0 then
            begin
              writeln('Error: Note Value is 0');
              halt(1);
            end;
            if not noteoff then
              write_bits(notetbl_pos(sampnr, note), bitsneeded[pitch_table_length[sampnr]]);


          end; { if sampnr_table_len[track] > 0 }


        end; { if not skip }


        skip_onlyvol:
        (* Volume *)
          if not noteoff then
          write_bits(volumetbl_pos(track, volconvert[volume]), bitsneeded[volume_table_len[track]]);


      end; (* if event *)


    end; (* for track *)

    skip2:

  end; (* for row *)



  (* Bitwrite-Position mit 0 auffuellen fuer glatte Adresse zum naechsten Pattern *)
  if bitwrite_position mod 8 > 0 then (* wenn ungerade *)
    write_bits(0, 8 - bitwrite_position mod 8); (* restliche 0 Bits anhaengen *)
  writeln(logfile, 'Pattern ', pattern, ' ', bitwrite_position div 8 - lastpos_patdata, ' bytes');

end;


procedure write_zsm;
  var
    i, ii: byte;
    zsmfile: file;
    dmf_smpsize: longint;
    next_dmf_sample_seek: longint;
begin
  assign(zsmfile, zsmname+'.zsm');
  rewrite(zsmfile, 1);


  (* Header fllen *)
  zsmheader.kennung := $4D54535A;
  zsmheader.version := 1;
  fillchar(zsmheader.titel[0], 20, 0);
  for i := 0 to 19 do
    zsmheader.titel[i] := dmf_songname[i+1];
  fillchar(zsmheader.composer[0], 20, 0);
  for i := 0 to 19 do
    zsmheader.composer[i] := dmf_composer[i+1];
  zsmheader.sample_entries := dmf_sample_entries;
  zsmheader.sequencer_entries := dmf_sequ_length;
  zsmheader.sequencer_loopstart := dmf_sequ_loopstart + 1;
  zsmheader.sequencer_loopend := dmf_sequ_loopend + 1;
  zsmheader.patterndata_size := bitwrite_position div 8;{patterndata_count;}

  (* Header schreiben *)
  blockwrite(zsmfile, zsmheader, sizeof(zsmheader));



  (* Sample-Infos schreiben, ohne Vol und C-3 *)
  for i := 1 to dmf_sample_entries do
    blockwrite(zsmfile, smpinfo[i], 5); (* aber mit Schleife und Marker *)


  (* Pitch-Tabellen schreiben *)
  blockwrite(zsmfile, pitch_table_length, dmf_sample_entries);
  for i := 1 to dmf_sample_entries do
    blockwrite(zsmfile, pitch_tables[i, 0], pitch_table_length[i]*3);

  (* Sequencer *)

  blockwrite(zsmfile, sequencer, zsmheader.sequencer_entries * 2);

  (* Pattern Daten *)
  blockwrite(zsmfile, patterns_pointer^, bitwrite_position div 8);


  (* Sample-Daten *)
  next_dmf_sample_seek := dmf_samplestream_seek;
  for i := 1 to zsmheader.sample_entries do
  begin
    seek(dmf_f, next_dmf_sample_seek);
    blockread(dmf_f, dmf_smpsize, 4);
    if dmf_smpsize <> smpinfo[i].length then
    begin
      writeln('Warning: Sample sizes do not match DMF vs. ZSM');
      writeln('Sample #', i, '  DMF: ', dmf_smpsize, '  ZSM: ', smpinfo[i].length);
    end;
    for ii := 1 to smpinfo[i].length div 2048 do
    begin
      blockread(dmf_f, buffer^, 2048);
      blockwrite(zsmfile, buffer^, 2048);
    end;
    blockread(dmf_f, buffer^, smpinfo[i].length mod 2048);
    blockwrite(zsmfile, buffer^, smpinfo[i].length mod 2048);
    inc(next_dmf_sample_seek, dmf_smpsize+4);
  end;



  close(zsmfile);

end;

procedure make_sequencer;
  var i: word;
begin

  for i := 1 to dmf_sequ_length do
    sequencer[i] := pattern_addr[dmf_sequencer[i] + 1];

end;

  var
    i: word;
    f: file;
begin

  assign(logfile, 'logfile.txt');
  rewrite(logfile);

  fill_freqtable;
  fill_bitsneeded_table;
  fill_volconvert;

  if paramstr(1) = '' then
  begin
    write('DMF (ohne Erw.): ');
    readln(zsmname);
  end
  else
    zsmname := paramstr(1);
  open_dmf(zsmname);

  load_dmf_infos;

  flag_used_patterns;

  make_pitch_tables;

  make_periods;

  make_sample_infos;

  set_up_patternstream;

  for i := 1 to dmf_pattentries do
  if pattern_used[i] then encode_pattern(i);

  writeln(logfile, 'Total Pattern Data: ', bitwrite_position div 8, ' byte');

  make_sequencer;

  getmem(buffer, 2048);

  write_zsm;

  close_dmf;
  close(logfile);
end.